#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include "dolgozat.h"


/* Struktúra, amely két dolgozat hasonlóságát tárolja pontszámban. */
typedef struct Hasonlosag {
    Dolgozat *d1, *d2;                  /* melyik dolgozatok */
    double mennyire_d1, mennyire_d2;
} Hasonlosag;


/* Beolvassa a megadott nevű vezérlőfájlt, és felépíti
 * a listát, amelyben a dolgozatok vannak. Igazzal tér vissza,
 * ha minden rendben volt. */
int vezerlofajl_beolvas(char *vezerlofajl, Dolgozat **pdolgozatok)
{
    char fajlnev[33];
    char nev[45];
    FILE *fp;
    int ret;

    fp=fopen(vezerlofajl, "rt");
    if (fp==NULL) {
        fprintf(stderr, "Nem lehet megnyitni a vezérlőfájlt: %s\n", vezerlofajl);
        return 0;
    }

    ret=fscanf(fp, "%s %[^\n]", fajlnev, nev);
    while (ret==2) {
        Dolgozat *uj=uj_dolgozat(fajlnev, nev);
        uj->kov=*pdolgozatok;
        *pdolgozatok=uj;
        ret=fscanf(fp, "%s %[^\n]", fajlnev, nev);
    }
    fclose(fp);
    if (ret!=EOF) {
        fprintf(stderr, "Hiba a vezérlőfájl olvasása közben: %s\n", vezerlofajl);
        return 0;
    }
    return 1;
}


/* Igazzal tér vissza, ha szót elválasztó karaktert kap.
 * Ez szóköz (enter, tabulátor) vagy írásjel. */
int elvalaszto(char c)
{
    return isspace(c) || ispunct(c);
}


/* Beolvas egy szót az adott fájlból.
 * A szó végén szóköz vagy írásjel van. Az elejéről eldobja ezeket.
 * Igazzal tér vissza, ha beolvasott valami értékelhetőt.
 * Legfeljebb SZOHOSSZ karaktert olvas; annál hosszabb szavaknál
 * a többi karaktert eldobja. Így maximum SZOHOSSZ+1 bájtot ír
 * a tömbbe (lezáró nullával együtt.) */
int szot_beolvas(FILE *fp, char *szo)
{
    int c, i;

    while ((c=fgetc(fp))!=EOF && elvalaszto(c))
        ; /* üres */
    if (c==EOF)
        return 0;
    i=0;
    szo[i++]=c;     /* első karakter */
    while ((c=fgetc(fp))!=EOF && !elvalaszto(c))
        if (i<SZOHOSSZ)
            szo[i++]=c;
    szo[i]='\0';

    return i!=0;
}


/* Dolgozatok beolvasása.
 * Végigmegy a dolgozatok listán, és mindegyik megadott fájlt beolvassa.
 * Igazzal tér vissza, ha minden rendben volt. */
int dolgozatok_beolvas(Dolgozat *dolgozatok)
{
    Dolgozat *iter;
    int fajlok=0, kifejezesek=0, meretek=0;

    for (iter=dolgozatok; iter!=NULL; iter=iter->kov) {
        FILE *fp;
        char szo[SZOHOSSZ], elozoszo[SZOHOSSZ]="";

        fp=fopen(iter->fajlnev, "rt");
        if (fp==NULL) {
            fprintf(stderr, "Nem lehet beolvasni a(z) %s fajlt!\n", iter->fajlnev);
            return 0;
        }
        /* egyesével olvassa a szavakat */
        while (szot_beolvas(fp, szo)) {
            char kifejezes[2*SZOHOSSZ];

            /* kifejezés = előző szó + mostani szó */
            strcpy(kifejezes, elozoszo);
            strcat(kifejezes, szo);
            halmaz_betesz(&iter->kifejezesek, kifejezes);
            kifejezesek++;

            /* következő iterációhoz */
            strcpy(elozoszo, szo);
        }
        fclose(fp);
        fajlok++;
        meretek+=halmaz_meret(iter->kifejezesek);
    }

    printf("%d fajl, %d kifejezes beolvasva.\n"
           "A halmazok osszesen %d elemuek.\n",
           fajlok, kifejezesek, meretek);

    return 1;
}


/* Összehasonlít két dolgozatot, és a hasonlítás
 * eredményével tér vissza (amely egy teljesen
 * kitöltött Hasonlóság struktúra.) */
Hasonlosag hasonlit(Dolgozat *d1, Dolgozat *d2)
{
    Hasonlosag h;
    int meret = halmaz_metszet_meret(d1->kifejezesek, d2->kifejezesek);

    h.d1=d1;
    h.d2=d2;
    h.mennyire_d1 = meret / (double) halmaz_meret(d1->kifejezesek);
    h.mennyire_d2 = meret / (double) halmaz_meret(d2->kifejezesek);

    return h;
}


/* Végigmegy az összes dolgozaton, és páronként összehasonlítja őket.
 * A hasonlóságok tömbbe beteszi az így kapott eredményeket, pontosan
 * n*(n-1) darabot, ahol n a dolgozatok lista mérete.
 * Az összehasonlítás nem szimmetrikus! Erről bővebben a részletes
 * dokumentációban. */
void hasonlit_osszes(Dolgozat *dolgozatok, Hasonlosag hasonlosagok[])
{
    Dolgozat *iter1, *iter2;
    int i=0;

    /* mindegyiket mindegyikkel - de paronkent csak egyszer */
    for (iter1=dolgozatok; iter1!=NULL; iter1=iter1->kov)
        for (iter2=iter1->kov; iter2!=NULL; iter2=iter2->kov)
            hasonlosagok[i++]=hasonlit(iter1, iter2);
}


/* Visszatér a két szám közül a nagyobbikkal. */
double max(double a, double b)
{
    return a>b ? a : b;
}


/* Rendező függvény a qsort()-tal való használathoz.
 * A két dolgozat hasonlóságának átlaga szerint csökkenő sorrendbe. */
int osszkep_szerint(void const *egyik, void const *masik)
{
    Hasonlosag *h1=(Hasonlosag*) egyik;
    Hasonlosag *h2=(Hasonlosag*) masik;
    double osszkep_h1 = max(h1->mennyire_d1, h1->mennyire_d2);
    double osszkep_h2 = max(h2->mennyire_d1, h2->mennyire_d2);
    if (osszkep_h1 < osszkep_h2)
        return 1;
    if (osszkep_h1 > osszkep_h2)
        return -1;
    return 0;
}


/* Főprogram */
int main(int argc, char **argv)
{
    Dolgozat *dolgozatok = NULL;
    Hasonlosag *hasonlosagok;
    int dolgszam, hasszam, i, ok;

    if (argc<2 || argc>3) {
        printf("Plagiumdetektor\n\n"
               "Hasznalat: %s <vezerlofajl> [kimenetifajl]\n\n"
               "A vezerlofajl formatuma soronkent:\n"
               "  fajlnev szerzo_neve\n"
               "A fajlnev nem tartalmazhat szokozoket, a szerzo neve viszont igen.\n", argv[0]);
        return 3;
    }

    /* beolvasás */
    ok=vezerlofajl_beolvas(argv[1], &dolgozatok);
    if (!ok)
        return 1;

    ok=dolgozatok_beolvas(dolgozatok);
    if (!ok)
        return 2;

    /* feldolgozás */
    dolgszam=dolgozatok_meret(dolgozatok);
    hasszam=dolgszam*(dolgszam-1)/2;
    hasonlosagok=(Hasonlosag*) malloc(hasszam*sizeof(Hasonlosag));
    hasonlit_osszes(dolgozatok, hasonlosagok);
    qsort(hasonlosagok, hasszam, sizeof(hasonlosagok[0]), osszkep_szerint);

    /* hasonlóságok a szabványos kimenetre */
    printf("\n20 legerosebb hasonlosag:\n");
    for (i=0; i<20; ++i)
        printf("%s (%s) <-> %s (%s), %.2f%%\n", hasonlosagok[i].d1->nev, hasonlosagok[i].d1->fajlnev,
                                                hasonlosagok[i].d2->nev, hasonlosagok[i].d2->fajlnev,
                                                max(hasonlosagok[i].mennyire_d1, hasonlosagok[i].mennyire_d2)*100);

    /* és ha kérte a felhasználó, fájlba */
    if (argv[2]!=NULL) {
        FILE *fp=fopen(argv[2], "wt");
        if (fp==NULL) {
            fprintf(stderr, "Nem tudom megnyitni a kimeneti fájlt: %s\n", argv[2]);
            return 4;
        }
        for (i=0; i<hasszam; ++i)
            fprintf(fp, "%.2f;%s;%s;%s;%s\n", max(hasonlosagok[i].mennyire_d1, hasonlosagok[i].mennyire_d2)*100,
                                               hasonlosagok[i].d1->nev, hasonlosagok[i].d1->fajlnev,
                                               hasonlosagok[i].d2->nev, hasonlosagok[i].d2->fajlnev);
        fclose(fp);
    }

    /* felszabadítások */
    free(hasonlosagok);
    dolgozatok_felszabadit(dolgozatok);

    return 0;
}
